<?php
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Headers: *");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

require '../../config.php';
header('Content-Type: application/json');

$connect = getDatabaseConnection();
if (!$connect) {
    echo json_encode(['status' => 'error', 'message' => 'Database connection failed']);
    exit;
}

// Ambil data dari POST
$tanggal_absensi = $_POST['tanggal_absensi'] ?? null;
$id_pegawai = $_POST['id_pegawai'] ?? null;
$Masuk_LatLong = $_POST['Masuk_LatLong'] ?? null;
$Masuk_Alamat = $_POST['Masuk_Alamat'] ?? null;
$status_kehadiran = $_POST['status_kehadiran'] ?? null;
$keterangan_absensi = $_POST['keterangan_absensi'] ?? null;
$jam_masuk = $_POST['jam_masuk'] ?? null;
$validasi = filter_var($_POST['validasi'] ?? false, FILTER_VALIDATE_BOOLEAN);

// Validasi input
if (!$tanggal_absensi || !$id_pegawai || !$status_kehadiran || !$jam_masuk) {
    echo json_encode(['status' => 'error', 'message' => 'Missing required fields']);
    exit;
}
// 🔒 CEK KECAMATAN CABANG SESUAI
// Ambil id_cabang milik pegawai
$getPegawaiCabang = $connect->prepare("SELECT id_cabang FROM t_pegawai WHERE id_pegawai = ?");
$getPegawaiCabang->bind_param("s", $id_pegawai);
$getPegawaiCabang->execute();
$resPegawai = $getPegawaiCabang->get_result();
if ($resPegawai->num_rows === 0) {
    echo json_encode(['status' => 'error', 'message' => 'Data pegawai tidak ditemukan']);
    exit;
}
$idCabangPegawai = $resPegawai->fetch_assoc()['id_cabang'];

// Ambil lokasi terdekat dari posisi yang dikirim (lat,lng)
list($lat, $lng) = explode(',', $Masuk_LatLong);
$getLokasi = $connect->prepare("
    SELECT l.id_cabang, c.nama_cabang AS name, l.lat, l.lng, l.radius,
        ( 6371 * ACOS( 
            COS(RADIANS(?)) * COS(RADIANS(l.lat)) *
            COS(RADIANS(l.lng) - RADIANS(?)) + SIN(RADIANS(?)) * SIN(RADIANS(l.lat))
        ) ) * 1000 AS distance
    FROM t_locations l
    JOIN t_cabang c ON l.id_cabang = c.id_cabang
    HAVING distance <= l.radius
    ORDER BY distance ASC
    LIMIT 1
");
$getLokasi->bind_param("ddd", $lat, $lng, $lat);
$getLokasi->execute();
$resLokasi = $getLokasi->get_result();

if ($resLokasi->num_rows === 0) {
    echo json_encode(['status' => 'error', 'message' => 'Lokasi Anda tidak sesuai radius lokasi kerja manapun']);
    exit;
}

$dataLokasi = $resLokasi->fetch_assoc();
if ((int)$dataLokasi['id_cabang'] !== (int)$idCabangPegawai) {
    echo json_encode(['status' => 'error', 'message' => 'Anda tidak absen di lokasi cabang yang sesuai. Lokasi ini milik cabang: ' . $dataLokasi['name']]);
    exit;
}
// Cek hari libur
$cekLibur = $connect->prepare("SELECT * FROM t_hari_libur WHERE tgl_libur = ?");
$cekLibur->bind_param("s", $tanggal_absensi);
$cekLibur->execute();
$isHariLibur = $cekLibur->get_result()->num_rows > 0;

if ($isHariLibur) {
    $cekShiftLibur = $connect->prepare("SELECT 1 FROM t_shift_jadwal WHERE id_pegawai = ? AND tanggal = ? LIMIT 1");
    $cekShiftLibur->bind_param("ss", $id_pegawai, $tanggal_absensi);
    $cekShiftLibur->execute();
    $hasShift = $cekShiftLibur->get_result()->num_rows > 0;

    if (!$hasShift) {
        echo json_encode(['status' => 'error', 'message' => 'Hari ini adalah hari libur. Anda tidak dijadwalkan shift, sehingga tidak bisa absen.']);
        exit;
    }
}

// Cek duplikat absen
$cekAbsen = $connect->prepare("SELECT 1 FROM t_absensi WHERE tanggal_absensi = ? AND id_pegawai = ?");
$cekAbsen->bind_param("ss", $tanggal_absensi, $id_pegawai);
$cekAbsen->execute();
if ($cekAbsen->get_result()->num_rows > 0) {
    echo json_encode(['status' => 'error', 'message' => 'Maaf, Anda sudah melakukan absen masuk hari ini']);
    exit;
}

// Cek shift
$cekShift = $connect->prepare("
    SELECT t.jam_mulai, t.jam_selesai FROM t_shift_jadwal j
    JOIN t_shift_tipe t ON j.id_shift_tipe = t.id_shift_tipe
    WHERE j.id_pegawai = ? AND j.tanggal = ? LIMIT 1
");
$cekShift->bind_param("ss", $id_pegawai, $tanggal_absensi);
$cekShift->execute();
$shiftResult = $cekShift->get_result();

$masukTime = DateTime::createFromFormat('H:i:s', $jam_masuk);
$telatMenit = 0;
$isLate = 0;

if ($shiftResult->num_rows > 0) {
    $shift = $shiftResult->fetch_assoc();
    $mulaiTime = DateTime::createFromFormat('H:i:s', $shift['jam_mulai']);
    $selesaiTime = DateTime::createFromFormat('H:i:s', $shift['jam_selesai']);
    $toleransiStart = clone $mulaiTime;
    $toleransiStart->modify('-20 minutes');

    if ($masukTime < $toleransiStart) {
        echo json_encode(['status' => 'error', 'message' => 'Anda belum bisa absen sekarang. Silakan mulai absen dari pukul ' . $toleransiStart->format('H:i')]);
        exit;
    }

    if ($masukTime > $mulaiTime) {
        $intervalTelat = $mulaiTime->diff($masukTime);
        $telatMenit = ($intervalTelat->h * 60) + $intervalTelat->i;

        // Toleransi 10 menit hanya untuk shift 07:30
        if ($mulaiTime->format('H:i:s') === '07:30:00' && $telatMenit <= 10) {
            // Masih dalam toleransi, dianggap tidak telat
            $telatMenit = 0;
            $isLate = 0;
        } else {
            // Di luar toleransi atau shift bukan 07:30
            if (!$validasi) {
                echo json_encode(['status' => 'error', 'message' => 'Anda telat ' . $telatMenit . ' menit']);
                exit;
            }
            $isLate = 1;
        }
    }

    if ($masukTime > $selesaiTime) {
        echo json_encode(['status' => 'error', 'message' => 'Waktu absen telah berakhir. Batas akhir absen adalah pukul ' . $selesaiTime->format('H:i')]);
        exit;
    }
} else {
    // Reguler (08:00 - 17:00)
    $jamMulai = DateTime::createFromFormat('H:i:s', '08:00:00');
    $jamSelesai = DateTime::createFromFormat('H:i:s', '17:00:00');
    $toleransiStart = clone $jamMulai;
    $toleransiStart->modify('-20 minutes');

    if ($masukTime < $toleransiStart) {
        echo json_encode(['status' => 'error', 'message' => 'Belum bisa absen. Absen mulai pukul ' . $toleransiStart->format('H:i')]);
        exit;
    }

    if ($masukTime > $jamSelesai) {
        echo json_encode(['status' => 'error', 'message' => 'Batas waktu absen berakhir pukul ' . $jamSelesai->format('H:i')]);
        exit;
    }

    if ($masukTime > $jamMulai) {
        $telatMenit = ($jamMulai->diff($masukTime)->h * 60) + $jamMulai->diff($masukTime)->i;
        if (!$validasi) {
            echo json_encode(['status' => 'error', 'message' => 'Anda telat ' . $telatMenit . ' menit', 'terlambat_menit' => $telatMenit]);
            exit;
        }
        $isLate = 1;
    }
}

// Upload gambar jika ada
$Masuk_Img = null;
$filepath = 'foto/';
if (!is_dir($filepath)) mkdir($filepath, 0755, true);

if (isset($_FILES['Masuk_Img'])) {
    $filename = time() . '_masuk.jpg';
    $fullpath = $filepath . $filename;
    if (move_uploaded_file($_FILES['Masuk_Img']['tmp_name'], $fullpath)) {
        $Masuk_Img = $filename;
    } else {
        echo json_encode(['status' => 'error', 'message' => 'Gagal upload gambar']);
        exit;
    }
}

// Simpan absen
$stmt = $connect->prepare("
    INSERT INTO t_absensi 
    (tanggal_absensi, id_pegawai, Masuk_LatLong, Masuk_Img, Masuk_Alamat, status_kehadiran, keterangan_absensi, jam_masuk, is_late, late_minutes)
    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
");
$stmt->bind_param("ssssssssii", $tanggal_absensi, $id_pegawai, $Masuk_LatLong, $Masuk_Img, $Masuk_Alamat, $status_kehadiran, $keterangan_absensi, $jam_masuk, $isLate, $telatMenit);

if ($stmt->execute()) {
    echo json_encode([
        'status' => 'success',
        'message' => 'Selamat, absen masuk berhasil',
        'filename' => $Masuk_Img,
        'is_late' => $isLate,
        'terlambat_menit' => $telatMenit
    ]);
} else {
    echo json_encode(['status' => 'error', 'message' => 'Gagal menyimpan absen']);
}

$stmt->close();
$connect->close();
